<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

// Get parameters from ZarinPal callback (via Iran server redirect)
$authority = $_GET['Authority'] ?? '';
$status = $_GET['Status'] ?? '';
$orderId = $_GET['order'] ?? '';

$pageTitle = 'نتیجه پرداخت';

// Log for debugging
error_log("Callback received - Authority: $authority, Status: $status, Order: $orderId");

// Find order by ID or authority
$conn = getDBConnection();
$order = null;

if ($orderId) {
    // Try to find by order ID first
    $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->bind_param("i", $orderId);
    $stmt->execute();
    $result = $stmt->get_result();
    $order = $result->fetch_assoc();
    $stmt->close();
}

// If not found by ID, try by authority
if (!$order && $authority) {
    $stmt = $conn->prepare("SELECT * FROM orders WHERE payment_authority = ?");
    $stmt->bind_param("s", $authority);
    $stmt->execute();
    $result = $stmt->get_result();
    $order = $result->fetch_assoc();
    $stmt->close();
}

$conn->close();

$paymentSuccess = false;
$refId = '';
$error = '';

if (!$order) {
    $error = 'سفارش یافت نشد. Order ID: ' . $orderId . ', Authority: ' . $authority;
    error_log("Order not found - Order ID: $orderId, Authority: $authority");
} elseif ($status !== 'OK') {
    $error = 'پرداخت توسط کاربر لغو شد';
    updateOrderStatus($order['id'], 'cancelled');
    error_log("Payment cancelled by user - Order ID: " . $order['id']);
} else {
    // Update authority if not set
    if (empty($order['payment_authority']) && $authority) {
        updateOrderStatus($order['id'], 'pending', ['authority' => $authority]);
        error_log("Updated authority for order " . $order['id'] . ": $authority");
    }
    // Verify payment with Iran server
    // Amount must be in Rials (Toman * 10)
    $verifyData = [
        'action' => 'verify_payment',
        'authority' => $authority,
        'amount' => (int)$order['price_irr'] * 10
    ];

    $ch = curl_init(IRAN_SERVER_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($verifyData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    error_log("Verify payment response - HTTP: $httpCode, Response: $response");

    if ($httpCode === 200 && $response) {
        $responseData = json_decode($response, true);
        
        if ($responseData && $responseData['status'] === 'success') {
            $refId = $responseData['ref_id'] ?? 'N/A';
            $paymentSuccess = true;
            
            error_log("Payment verified successfully - Order: " . $order['id'] . ", Ref ID: $refId");
            
            // Update order status
            updateOrderStatus($order['id'], 'paid', ['ref_id' => $refId]);
            
            // Check if this is a TOPUP order or regular order
            $orderType = $order['order_type'] ?? 'regular';
            $isTopup = ($orderType === 'topup');
            
            // Create eSIM or Topup via API
            try {
                $api = new eSIMAPI();
                
                if ($isTopup) {
                    // This is a TOPUP order
                    error_log("Processing TOPUP for order " . $order['id'] . ", package: " . $order['package_code']);
                    
                    // Get parent eSIM details
                    $parentOrderId = $order['parent_order_id'] ?? 0;
                    if (!$parentOrderId) {
                        throw new Exception("Parent order ID not found for topup");
                    }
                    
                    $conn = getDBConnection();
                    $stmt = $conn->prepare("SELECT iccid, esim_tran_no FROM orders WHERE id = ? AND status = 'paid'");
                    $stmt->bind_param("i", $parentOrderId);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $parentEsim = $result->fetch_assoc();
                    $stmt->close();
                    $conn->close();
                    
                    if (!$parentEsim || empty($parentEsim['esim_tran_no'])) {
                        throw new Exception("Parent eSIM not found or missing esimTranNo");
                    }
                    
                    // Perform topup
                    $transactionId = 'TOPUP_' . time() . '_' . $order['id'];
                    $esimResult = $api->topupESIM($parentEsim['esim_tran_no'], $order['package_code'], $transactionId);
                    
                    error_log("Topup API response: " . json_encode($esimResult));
                    
                    if ($esimResult && isset($esimResult['success']) && $esimResult['success'] === true) {
                        error_log("Topup successful for order " . $order['id']);
                        // Topup successful - no need to create new eSIM
                    } else {
                        $errorCode = $esimResult['errorCode'] ?? 'unknown';
                        $errorMsg = $esimResult['errorMsg'] ?? 'Unknown error';
                        error_log("Topup API returned error - Code: $errorCode, Message: $errorMsg");
                    }
                } else {
                    // Regular eSIM order
                    error_log("Creating eSIM for order " . $order['id'] . ", package: " . $order['package_code']);
                    
                    // Get current package price from API (to avoid "price expired" error)
                    $packageData = $api->getPackageByCode($order['package_code']);
                    
                    if ($packageData && isset($packageData['price'])) {
                        $currentPrice = $packageData['price']; // Already in cents
                        error_log("Current package price from API: $currentPrice cents");
                    } else {
                        // Fallback to stored price
                        $currentPrice = round($order['price_usd'] * 100);
                        error_log("Using stored price (API didn't return price): $currentPrice cents");
                    }
                    
                    $esimResult = $api->createESIMOrder($order['package_code'], 1, $currentPrice);
                    
                    error_log("eSIM API response: " . json_encode($esimResult));
                    
                    if ($esimResult && isset($esimResult['success']) && $esimResult['success'] === true) {
                        // Order created successfully, now query the eSIM details
                        $orderNo = $esimResult['obj']['orderNo'] ?? null;
                        
                        if ($orderNo) {
                            error_log("Order created successfully, orderNo: $orderNo. Querying eSIM details...");
                            
                            // Wait a moment for the eSIM to be allocated
                            sleep(2);
                            
                            // Query the eSIM details
                            $queryResult = $api->queryESIMByOrderNo($orderNo);
                            error_log("Query eSIM response: " . json_encode($queryResult));
                            
                            if ($queryResult && isset($queryResult['success']) && $queryResult['success'] === true) {
                                $esimList = $queryResult['obj']['esimList'] ?? [];
                                
                                if (!empty($esimList)) {
                                    $esim = $esimList[0]; // Get first eSIM
                                    
                                    $esimData = [
                                        'esim_id' => $esim['esimTranNo'] ?? '',
                                        'iccid' => $esim['iccid'] ?? '',
                                        'qr_code' => $esim['qrCodeUrl'] ?? '',
                                        'activation_code' => $esim['ac'] ?? ''
                                    ];
                                    
                                    updateOrderWithESIM($order['id'], $esimData);
                                    error_log("eSIM created successfully for order " . $order['id'] . ", ICCID: " . $esimData['iccid']);
                                } else {
                                    error_log("eSIM list is empty for orderNo: $orderNo");
                                }
                            } else {
                                error_log("Failed to query eSIM details for orderNo: $orderNo");
                            }
                        } else {
                            error_log("OrderNo not found in API response");
                        }
                    } else {
                        $errorCode = $esimResult['errorCode'] ?? 'unknown';
                        $errorMsg = $esimResult['errorMsg'] ?? 'Unknown error';
                        error_log("eSIM API returned error - Code: $errorCode, Message: $errorMsg");
                    }
                }
            } catch (Exception $e) {
                // Log error but don't fail the payment
                error_log('eSIM creation failed: ' . $e->getMessage());
            }
        } else {
            $error = $responseData['message'] ?? 'خطا در تایید پرداخت';
            updateOrderStatus($order['id'], 'failed');
            error_log("Payment verification failed - Order: " . $order['id'] . ", Error: $error");
        }
    } else {
        $error = 'خطا در اتصال به سرور تایید پرداخت. HTTP: ' . $httpCode;
        if ($curlError) {
            $error .= ', cURL Error: ' . $curlError;
        }
        updateOrderStatus($order['id'], 'failed');
        error_log("Verify payment connection failed - HTTP: $httpCode, cURL Error: $curlError");
    }
}

include __DIR__ . '/includes/header.php';
?>

<style>
.result-container {
    max-width: 600px;
    margin: 50px auto;
    background: white;
    border-radius: 15px;
    padding: 40px;
    box-shadow: 0 5px 25px rgba(0, 0, 0, 0.1);
    text-align: center;
}

.result-icon {
    font-size: 100px;
    margin-bottom: 20px;
}

.result-icon.success {
    color: #28a745;
}

.result-icon.error {
    color: #dc3545;
}

.result-title {
    font-size: 28px;
    margin-bottom: 15px;
}

.result-title.success {
    color: #28a745;
}

.result-title.error {
    color: #dc3545;
}

.result-message {
    color: #666;
    font-size: 16px;
    line-height: 1.8;
    margin: 20px 0;
}

.order-details {
    background: #f8f9fa;
    padding: 25px;
    border-radius: 10px;
    margin: 30px 0;
    text-align: right;
}

.order-details h3 {
    color: #667eea;
    margin-bottom: 20px;
    text-align: center;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 12px 0;
    border-bottom: 1px solid #e0e0e0;
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-row strong {
    color: #999;
}

.detail-row span {
    color: #333;
    font-weight: 600;
}

.ref-id-box {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 20px;
    border-radius: 10px;
    margin: 20px 0;
}

.ref-id-box .label {
    font-size: 14px;
    opacity: 0.9;
}

.ref-id-box .value {
    font-size: 24px;
    font-weight: bold;
    margin-top: 10px;
}

.btn {
    display: inline-block;
    padding: 14px 35px;
    background: #667eea;
    color: white;
    text-decoration: none;
    border-radius: 8px;
    margin: 10px 5px;
    transition: background 0.3s;
    font-weight: bold;
}

.btn:hover {
    background: #5568d3;
}

.btn-success {
    background: #28a745;
}

.btn-success:hover {
    background: #218838;
}

.error-box {
    background: #fee;
    border: 1px solid #fcc;
    color: #c33;
    padding: 20px;
    border-radius: 8px;
    margin: 20px 0;
}
</style>

<div class="container">
    <div class="result-container">
        <?php if ($paymentSuccess): ?>
            <div class="result-icon success">
                <i class="fas fa-check-circle"></i>
            </div>
            <?php 
            $orderType = $order['order_type'] ?? 'regular';
            $isTopup = ($orderType === 'topup');
            ?>
            <h1 class="result-title success"><?php echo $isTopup ? 'شارژ مجدد موفق!' : 'پرداخت موفق!'; ?></h1>
            <p class="result-message">
                <?php if ($isTopup): ?>
                    پرداخت شما با موفقیت انجام شد. حجم و اعتبار به eSIM شما اضافه شده است.
                <?php else: ?>
                    پرداخت شما با موفقیت انجام شد. eSIM شما آماده است و به ایمیل شما ارسال خواهد شد.
                <?php endif; ?>
            </p>

            <div class="ref-id-box">
                <div class="label">شماره پیگیری پرداخت</div>
                <div class="value"><?php echo htmlspecialchars($refId); ?></div>
            </div>

            <div class="order-details">
                <h3><i class="fas fa-file-invoice"></i> جزئیات سفارش</h3>
                <div class="detail-row">
                    <strong>شماره سفارش:</strong>
                    <span>#<?php echo $order['id']; ?></span>
                </div>
                <div class="detail-row">
                    <strong>بسته:</strong>
                    <span><?php echo htmlspecialchars($order['package_name']); ?></span>
                </div>
                <div class="detail-row">
                    <strong>کشور:</strong>
                    <span><?php echo htmlspecialchars($order['country_name']); ?></span>
                </div>
                <?php if (!empty($order['data_amount'])): ?>
                <div class="detail-row">
                    <strong>حجم:</strong>
                    <span><?php echo htmlspecialchars($order['data_amount']); ?></span>
                </div>
                <?php endif; ?>
                <?php if (!empty($order['validity_days'])): ?>
                <div class="detail-row">
                    <strong>مدت:</strong>
                    <span><?php echo intval($order['validity_days']); ?> روز</span>
                </div>
                <?php endif; ?>
                <div class="detail-row">
                    <strong>مبلغ پرداختی:</strong>
                    <span><?php echo formatPrice($order['price_irr']); ?> تومان</span>
                </div>
            </div>

            <div style="margin-top: 30px;">
                <?php if ($isTopup): ?>
                    <?php 
                    // For topup, link to parent eSIM
                    $parentOrderId = $order['parent_order_id'] ?? 0;
                    ?>
                    <a href="/esim-details.php?id=<?php echo $parentOrderId; ?>" class="btn btn-success">
                        <i class="fas fa-sim-card"></i>
                        مشاهده eSIM شارژ شده
                    </a>
                    <a href="/user/dashboard.php" class="btn">
                        <i class="fas fa-list"></i>
                        لیست سفارشات من
                    </a>
                <?php else: ?>
                    <a href="/esim-details.php?id=<?php echo $order['id']; ?>" class="btn btn-success">
                        <i class="fas fa-sim-card"></i>
                        مشاهده eSIM خریداری شده
                    </a>
                    <a href="/my-esims.php" class="btn">
                        <i class="fas fa-list"></i>
                        لیست eSIM های من
                    </a>
                <?php endif; ?>
                <a href="/" class="btn">
                    <i class="fas fa-home"></i>
                    بازگشت به خانه
                </a>
            </div>

        <?php else: ?>
            <div class="result-icon error">
                <i class="fas fa-times-circle"></i>
            </div>
            <h1 class="result-title error">پرداخت ناموفق</h1>
            
            <div class="error-box">
                <i class="fas fa-exclamation-triangle"></i>
                <?php echo htmlspecialchars($error); ?>
            </div>

            <p class="result-message">
                متأسفانه پرداخت شما انجام نشد. در صورت کسر مبلغ از حساب، طی 72 ساعت به حساب شما بازگردانده می‌شود.
            </p>

            <?php if ($order): ?>
                <div style="margin-top: 30px;">
                    <a href="/checkout.php?package=<?php echo urlencode($order['package_code']); ?>&country=<?php echo urlencode($order['country_code']); ?>" class="btn btn-success">
                        <i class="fas fa-redo"></i>
                        تلاش مجدد
                    </a>
                    <a href="/" class="btn">
                        <i class="fas fa-home"></i>
                        بازگشت به خانه
                    </a>
                </div>
            <?php else: ?>
                <div style="margin-top: 30px;">
                    <a href="/" class="btn">
                        <i class="fas fa-home"></i>
                        بازگشت به خانه
                    </a>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>

